<?php
// adminhub/tools/add_module.php
// Admin aracı: Modül Ekleme ve Silme
// ÖZELLİK: Yeni modülü, mevcut "module_" sütunlarının en sonuncusunun hemen yanına ekler.

require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../modules_registry.php';

if (!is_admin()) {
    http_response_code(403);
    echo "Yetkisiz";
    exit;
}

$db  = hub_db();
$err = null;
$ok  = null;

// SİLİNMESİ YASAK OLAN ÇEKİRDEK MODÜLLER
$protectedModules = ['invoices', 'customers', 'products', 'reports'];

/**
 * Yardımcı Fonksiyon: Veritabanındaki 'module_' ile başlayan gerçek modül listesini çeker.
 */
function get_db_modules_list(PDO $pdo): array {
    $mods = [];
    try {
        $q = $pdo->query("SHOW COLUMNS FROM tenants LIKE 'module_%'");
        while($r = $q->fetch(PDO::FETCH_ASSOC)) {
            $mods[] = substr($r['Field'], 7); 
        }
    } catch (Throwable $e) {}
    return $mods;
}

/**
 * Yardımcı Fonksiyon: JSON Registry dosyasını DB'ye bakarak yeniden oluşturur.
 */
function sync_registry_json(PDO $pdo): bool {
    try {
        $finalDbModules = get_db_modules_list($pdo);
        $newRegistry = [];

        foreach ($finalDbModules as $m) {
            $newRegistry[$m] = [
                'label'         => strtoupper(str_replace('_', ' ', $m)),
                'tenant_column' => 'module_' . $m,
                'config_key'    => 'MODULE_' . strtoupper($m),
                'api_field'     => $m
            ];
        }

        $jsonFile = __DIR__ . '/../includes/modules_registry.json';
        return (bool)file_put_contents($jsonFile, json_encode($newRegistry, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    } catch (Throwable $e) {
        return false;
    }
}

// POST İŞLEMLERİ
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!csrf_check($_POST['csrf'] ?? '')) {
        $err = 'Güvenlik doğrulaması başarısız.';
    } else {
        $action = $_POST['action'] ?? '';
        $moduleRaw = (string)($_POST['module'] ?? '');
        $module    = strtolower(trim($moduleRaw));

        // --- MODÜL EKLEME ---
        if ($action === 'add') {
            if ($module === '' || !preg_match('/^[a-z0-9_]+$/', $module)) {
                $err = 'Geçersiz modül adı.';
            } else {
                $existing = get_db_modules_list($db);
                if (in_array($module, $existing, true)) {
                    $err = "Bu modül zaten mevcut: '{$module}'";
                } else {
                    try {
                        // 1. Tenants Kolonu Ekle (SIRALI)
                        // Önce en son 'module_' ile başlayan kolonu bulalım
                        $lastModStmt = $db->query("
                            SELECT COLUMN_NAME 
                            FROM INFORMATION_SCHEMA.COLUMNS 
                            WHERE TABLE_SCHEMA = DATABASE() 
                              AND TABLE_NAME = 'tenants' 
                              AND COLUMN_NAME LIKE 'module_%' 
                            ORDER BY ORDINAL_POSITION DESC 
                            LIMIT 1
                        ");
                        $lastModCol = $lastModStmt->fetchColumn();

                        $col = 'module_' . $module;
                        $sql = "ALTER TABLE tenants ADD COLUMN `{$col}` TINYINT(1) NOT NULL DEFAULT 0";
                        
                        // Eğer başka bir module_ kolonu varsa, onun arkasına ekle
                        if ($lastModCol) {
                            $sql .= " AFTER `{$lastModCol}`";
                        }
                        
                        $db->exec($sql);

                        // 2. Tenant Verilerini Ekle
                        $tenants = $db->query("SELECT id,db_host,db_port,db_name,db_user,db_pass FROM tenants")->fetchAll(PDO::FETCH_ASSOC);
                        $appKey  = 'MODULE_' . strtoupper($module);
                        $flagKey = $module;

                        foreach ($tenants as $t) {
                            try {
                                $dsn = "mysql:host={$t['db_host']};port={$t['db_port']};dbname={$t['db_name']};charset=utf8mb4";
                                $pdo = new PDO($dsn, $t['db_user'], $t['db_pass'], [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);
                                
                                $pdo->prepare("INSERT INTO app_settings (k, v) VALUES (?, '0') ON DUPLICATE KEY UPDATE v=v")->execute([$appKey]);
                                
                                $stf = $pdo->prepare("SELECT COUNT(*) FROM module_flags WHERE module_key = ?");
                                $stf->execute([$flagKey]);
                                if (!$stf->fetchColumn()) {
                                    $pdo->prepare("INSERT INTO module_flags (module_key, is_enabled) VALUES (?, 0)")->execute([$flagKey]);
                                }
                            } catch (Throwable $e) { /* Loglanabilir */ }
                        }

                        // 3. Sync JSON
                        if (sync_registry_json($db)) {
                            $ok = "Modül EKLENDİ: '{$module}'. Veritabanı ve JSON dosyası güncellendi.";
                        } else {
                            $ok = "Modül eklendi ancak JSON dosyası yazılamadı.";
                        }
                    } catch (Throwable $e) {
                        $err = 'Ekleme başarısız: ' . $e->getMessage();
                    }
                }
            }
        } 
        
        // --- MODÜL SİLME ---
        elseif ($action === 'delete') {
            if (in_array($module, $protectedModules)) {
                $err = "Bu modül çekirdek modüldür ve silinemez: '{$module}'";
            } else {
                try {
                    // 1. Tenants Kolonunu Sil
                    $col = 'module_' . $module;
                    $check = $db->prepare("SHOW COLUMNS FROM tenants LIKE ?");
                    $check->execute([$col]);
                    if ($check->rowCount() > 0) {
                        $db->exec("ALTER TABLE tenants DROP COLUMN `{$col}`");
                    }

                    // 2. Tenant Verilerini Temizle
                    $tenants = $db->query("SELECT id,db_host,db_port,db_name,db_user,db_pass FROM tenants")->fetchAll(PDO::FETCH_ASSOC);
                    $appKey  = 'MODULE_' . strtoupper($module);
                    $flagKey = $module;

                    foreach ($tenants as $t) {
                        try {
                            $dsn = "mysql:host={$t['db_host']};port={$t['db_port']};dbname={$t['db_name']};charset=utf8mb4";
                            $pdo = new PDO($dsn, $t['db_user'], $t['db_pass'], [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);
                            
                            $pdo->prepare("DELETE FROM app_settings WHERE k = ?")->execute([$appKey]);
                            $pdo->prepare("DELETE FROM module_flags WHERE module_key = ?")->execute([$flagKey]);

                        } catch (Throwable $e) { /* Loglanabilir */ }
                    }

                    // 3. Sync JSON
                    if (sync_registry_json($db)) {
                        $ok = "Modül SİLİNDİ: '{$module}'. Veritabanından ve tenantlardan temizlendi.";
                    } else {
                        $ok = "Modül silindi ancak JSON dosyası güncellenemedi.";
                    }

                } catch (Throwable $e) {
                    $err = 'Silme başarısız: ' . $e->getMessage();
                }
            }
        }
    }
}

$displayModules = get_db_modules_list($db);
?>
<?php include __DIR__ . '/../includes/header.php'; ?>

<div class="container py-4">
  <div class="row mb-3">
    <div class="col-12 col-lg-10 mx-auto">
      <div class="d-flex flex-wrap justify-content-between align-items-center gap-2">
        <div>
          <div class="small text-muted mb-1">AdminHub &raquo; Modül Yönetimi</div>
          <h3 class="mb-0">Modül Ekle / Kaldır</h3>
        </div>
        <div>
          <a class="btn btn-light border" href="<?= h(base_url('tenants/index.php')) ?>">&larr; Tenant Listesi</a>
        </div>
      </div>
    </div>
  </div>

  <div class="row mb-3">
    <div class="col-12 col-lg-10 mx-auto">
      <?php if ($err): ?>
        <div class="alert alert-danger alert-dismissible fade show"><?= h($err) ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
      <?php endif; ?>
      <?php if ($ok): ?>
        <div class="alert alert-success alert-dismissible fade show"><?= nl2br(h($ok)) ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>
      <?php endif; ?>
    </div>
  </div>

  <div class="row g-4">
    <div class="col-12 col-lg-5 mx-auto">
      <div class="card shadow-sm border-0 mb-3">
        <div class="card-header bg-white border-0 pb-0">
          <h5 class="card-title mb-1">Yeni Modül Ekle</h5>
          <p class="text-muted small mb-0">Veritabanına yeni bir modül kolonu ekler.</p>
        </div>
        <div class="card-body">
          <form method="post">
            <input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>">
            <input type="hidden" name="action" value="add">
            
            <div class="mb-3">
              <label class="form-label fw-semibold">Modül Anahtarı</label>
              <input name="module" class="form-control" placeholder="örnek: stok_takip" pattern="[a-z0-9_]+" required>
              <div class="form-text">Küçük harf, rakam ve alt çizgi.</div>
            </div>
            <button class="btn btn-primary w-100"><i class="bi bi-plus-lg"></i> Ekle ve Yapılandır</button>
          </form>
        </div>
      </div>
      
      <div class="card shadow-sm border-0">
        <div class="card-header bg-white border-0 pb-0"><h6>Otomatik İşlemler</h6></div>
        <div class="card-body pt-2">
          <ul class="small ps-3 mb-0">
            <li><strong>Sıralama:</strong> Yeni sütun, mevcut modül sütunlarının sonuncusunun yanına eklenir.</li>
            <li><strong>Ekleme:</strong> <code>tenants</code> tablosuna kolon ekler, JSON'u günceller.</li>
            <li><strong>Silme:</strong> Kolonu siler, tüm tenant veritabanlarından <code>app_settings</code> ve <code>module_flags</code> kayıtlarını temizler.</li>
          </ul>
        </div>
      </div>
    </div>

    <div class="col-12 col-lg-7">
      <div class="card shadow-sm border-0 mb-3">
        <div class="card-header bg-white border-0 pb-0">
          <h6 class="mb-1">Mevcut Modüller (<?= count($displayModules) ?>)</h6>
          <p class="text-muted small mb-0">Veritabanındaki aktif modül kolonları</p>
        </div>
        <div class="card-body p-0">
            <ul class="list-group list-group-flush">
                <?php if(empty($displayModules)): ?>
                    <li class="list-group-item text-center text-muted py-3">Modül bulunamadı.</li>
                <?php else: foreach($displayModules as $mod): ?>
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                        <div>
                            <strong><?= h($mod) ?></strong>
                            <div class="small text-muted">module_<?= h($mod) ?></div>
                        </div>
                        
                        <?php if(in_array($mod, $protectedModules)): ?>
                            <span class="badge bg-light text-dark border">
                                <i class="bi bi-shield-lock-fill"></i> Çekirdek
                            </span>
                        <?php else: ?>
                            <form method="post" onsubmit="return confirm('BU MODÜLÜ SİLMEK İSTEDİĞİNİZE EMİN MİSİNİZ?\n\nModül: <?=h($mod)?>\n\nBu işlem tenants tablosundan kolonu siler ve tüm müşterilerden ayarları temizler.');">
                                <input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="module" value="<?= h($mod) ?>">
                                <button class="btn btn-sm btn-outline-danger">
                                    <i class="bi bi-trash"></i> Sil
                                </button>
                            </form>
                        <?php endif; ?>
                    </li>
                <?php endforeach; endif; ?>
            </ul>
        </div>
      </div>
    </div>
  </div>
</div>

<?php include __DIR__ . '/../includes/footer.php'; ?>