<?php
require_once __DIR__ . '/../includes/auth.php';
// includes/auth.php -> includes/functions.php -> includes/modules_registry.php zinciri sayesinde fonksiyonlar hazır.

$active    = 'tenants';
$pageTitle = 'Tenants';
$db = hub_db();

/* ---- Modül registry'si ---- */
$moduleRegistry = hub_modules_registry(); // Global fonksiyon
$allModules     = array_keys($moduleRegistry);

/* ---- Filtre: module ---- */
$filterModule = isset($_GET['module']) ? strtolower(trim($_GET['module'])) : '';
if (!in_array($filterModule, $allModules, true)) {
    $filterModule = '';
}

/* ---- Sorgu ---- */
$where = "1=1";
$args  = [];
if ($filterModule !== '') {
    $cfg = $moduleRegistry[$filterModule] ?? null;
    $col = $cfg['tenant_column'] ?? ('module_' . $filterModule);
    // NOT: Filtre hala tenants tablosundaki kolon üzerinden çalışıyor.
    $where .= " AND {$col}=1";
}

$st = $db->prepare("SELECT * FROM tenants WHERE {$where} ORDER BY id DESC");
$st->execute($args);
$rows = $st->fetchAll();

/* ---- Kota ayarları ---- */
$c = hub_config();
$PLAN_LIMITS = [
  'free'  => isset($c['PLAN_LIMIT_FREE'])  ? (int)$c['PLAN_LIMIT_FREE']  : 5,
  'basic' => isset($c['PLAN_LIMIT_BASIC']) ? (int)$c['PLAN_LIMIT_BASIC'] : 100,
  'pro'   => null,
];

function tenant_invoice_count(array $tenant): ?int {
    try {
        $dsn = "mysql:host={$tenant['db_host']};port={$tenant['db_port']};dbname={$tenant['db_name']};charset=utf8mb4";
        $pdo = new PDO($dsn, $tenant['db_user'], $tenant['db_pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
        $cnt = $pdo->query("SELECT COUNT(*) FROM invoices")->fetchColumn();
        return ($cnt === false) ? 0 : (int)$cnt;
    } catch (Throwable $e) {
        return null;
    }
}

/* ---- Modül rozetleri: UZAK DB'DEN OKUYAN HALİ ---- */
function module_badges(array $tenantRow, string $currentFilter): string {
    $mods = hub_modules_registry();
    if (!$mods) return '';

    $defs = []; // $key => bool (on/off)

    // Uzak (tenant) DB bağlantısını kurmayı dene
    $pdo = null;
    $host = $tenantRow['db_host'] ?? '';
    $name = $tenantRow['db_name'] ?? '';
    $user = $tenantRow['db_user'] ?? '';
    if ($host && $name && $user) {
        try {
            $port = (int)($tenantRow['db_port'] ?? 3306);
            $pass = $tenantRow['db_pass'] ?? '';
            $dsn  = "mysql:host={$host};port={$port};dbname={$name};charset=utf8mb4";
            $pdo  = new PDO($dsn, $user, $pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ]);
        } catch (Throwable $e) {
            $pdo = null; // Uzak DB erişimi yoksa fallback kullanacağız
        }
    }

    foreach ($mods as $key => $cfg) {
        $on = false;

        if ($pdo) {
            try {
                // 1) Önce module_flags'tan oku
                $stmt = $pdo->prepare("SELECT is_enabled FROM module_flags WHERE module_key = ? LIMIT 1");
                $stmt->execute([$key]);
                $row = $stmt->fetch();

                if ($row && array_key_exists('is_enabled', $row)) {
                    $on = (int)$row['is_enabled'] ? true : false;
                } else {
                    // 2) Fallback: app_settings üzerinden oku
                    $configKey = $cfg['config_key'] ?? ('MODULE_' . strtoupper($key));
                    $stmt2 = $pdo->prepare("SELECT v FROM app_settings WHERE k = ? LIMIT 1");
                    $stmt2->execute([$configKey]);
                    $row2 = $stmt2->fetch();

                    if ($row2 && array_key_exists('v', $row2)) {
                        $on = ((int)$row2['v']) ? true : false;
                    } else {
                        // 3) Son fallback: yakın DB'deki tenants.module_... kolonu
                        $col = $cfg['tenant_column'] ?? null;
                        $on  = $col ? (bool)($tenantRow[$col] ?? 0) : false;
                    }
                }
            } catch (Throwable $e) {
                // Uzak sorgu patlarsa yine tenants.tablosuna düş
                $col = $cfg['tenant_column'] ?? null;
                $on  = $col ? (bool)($tenantRow[$col] ?? 0) : false;
            }
        } else {
            // Uzak DB yoksa tamamen tenants.module_* üzerinden karar ver
            $col = $cfg['tenant_column'] ?? null;
            $on  = $col ? (bool)($tenantRow[$col] ?? 0) : false;
        }

        $defs[$key] = $on;
    }

    $total  = count($defs);
    if ($total === 0) return '';
    $perRow = (int)ceil($total / 3);

    $html = '<div class="d-inline-block">';
    $i    = 0;

    foreach ($defs as $key => $on) {
        $cfg   = $mods[$key] ?? [];
        $cls   = $on ? 'bg-success' : 'bg-danger';
        $q     = http_build_query(['module' => $key]);
        $activeMark = ($currentFilter === $key) ? ' outline-light' : '';
        $label = $cfg['label'] ?? strtoupper($key);
        $title = ($on ? 'Açık: ' : 'Kapalı: ') . $label;

        $html .= '<a class="badge '.$cls.$activeMark.' me-1 mb-1" title="'.h($title).'" href="?'.$q.'">'
              .  h($label)
              .  '</a>';
        $i++;

        if (($i % $perRow === 0) && ($i < $total)) {
            $html .= '<br>';
        }
    }
    $html .= '</div>';
    return $html;
}

/* ---- Hızlı filtre butonları: Dynamic ---- */
function filter_pills(string $current): string {
    $mods = hub_modules_registry();
    $base = h(base_url('tenants/index.php'));

    $out  = '<div class="d-flex flex-wrap gap-2">';
    $out .= '<a class="btn btn-sm '.($current===''?'btn-primary':'btn-outline-primary').'" href="'.$base.'">Tümü</a>';

    foreach ($mods as $key => $cfg) {
        $sel   = ($current === $key) ? 'btn-primary' : 'btn-outline-primary';
        $label = $cfg['label'] ?? strtoupper($key);
        $out  .= '<a class="btn btn-sm '.$sel.'" href="'.$base.'?module='.$key.'">'.$label.'</a>';
    }

    $out .= '</div>';
    return $out;
}

include __DIR__ . '/../includes/header.php';
?>
<div class="d-flex justify-content-between align-items-center mb-3">
  <h3 class="mb-0">Tenants</h3>
  <div class="d-flex gap-2">
    <a class="btn btn-outline-primary" href="<?= h(base_url('tenants/bulk.php')) ?>"><i class="bi bi-layers"></i> Toplu</a>
    <a class="btn class btn-primary" href="<?= h(base_url('tenants/add.php')) ?>"><i class="bi bi-plus-circle"></i> Yeni Tenant</a>
  </div>
</div>

<div class="mb-3">
  <?= filter_pills($filterModule) ?>
</div>

<div class="card">
  <div class="card-body p-0">
    <div class="table-responsive">
      <table class="table table-striped table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th>#</th>
            <th>Ad</th>
            <th>Domain</th>
            <th>Yöntem</th>
            <th>Abonelik</th>
            <th>Fatura</th>
            <th>API</th>
            <th>Modüller</th>
            <th>Aktif</th>
            <th>Son Uygulama</th>
            <th class="text-end">İşlemler</th>
          </tr>
        </thead>
        <tbody>
        <?php foreach ($rows as $r): ?>
          <?php
            $method = $r['apply_method'] ?? 'db';
            $plan   = $r['subscription_type'] ?? 'free';
            $limit  = $PLAN_LIMITS[$plan] ?? null;

            $invCount = tenant_invoice_count($r);
            if ($invCount === null) {
                $usageHtml = '<span class="badge bg-secondary" title="Bağlantı/erişim hatası">N/A</span>';
            } else {
                $usageText = ($limit === null) ? ($invCount . '/&infin;') : ($invCount . '/' . (int)$limit);
                $usageHtml = '<span class="badge bg-light text-dark">'.$usageText.'</span>';
            }

            // API hücresi
            $apiUrl = trim((string)($r['api_base_url'] ?? ''));
            $apiKey = trim((string)($r['api_key'] ?? ''));
            if ($method === 'api') {
                if ($apiUrl !== '' && $apiKey !== '') {
                    $short = preg_replace('~^https?://~', '', $apiUrl);
                    if (mb_strlen($short, 'UTF-8') > 30) {
                        $short = mb_substr($short, 0, 27, 'UTF-8') . '…';
                    }
                    $apiCell = '<span class="badge bg-success" title="'.h($apiUrl).'">Bağlı</span>'
                             . '<small class="text-muted d-block">'.h($short).'</small>';
                } elseif ($apiUrl !== '' && $apiKey === '') {
                    $apiCell = '<span class="badge bg-warning text-dark" title="API Key eksik">Key yok</span>';
                } elseif ($apiUrl === '' && $apiKey !== '') {
                    $apiCell = '<span class="badge bg-warning text-dark" title="API Base URL eksik">URL yok</span>';
                } else {
                    $apiCell = '<span class="badge bg-secondary" title="API URL ve Key tanımsız">Tanımsız</span>';
                }
            } else {
                $apiCell = '<span class="badge bg-light text-dark" title="Uygulama yöntemi DB">DB</span>';
            }
          ?>
          <tr>
            <td><?= (int)$r['id'] ?></td>
            <td><?= h($r['name']) ?></td>
            <td><?= h($r['domain']) ?></td>
            <td><span class="badge bg-secondary"><?= h($method) ?></span></td>
            <td><?= h($plan) ?><?= !empty($r['subscription_end']) ? ' / '.h($r['subscription_end']) : '' ?></td>
            <td><?= $usageHtml ?></td>
            <td><?= $apiCell ?></td>
            <td><?= module_badges($r, $filterModule) ?></td>
            <td><?= (int)$r['is_active'] ? '<span class="badge bg-success">Aktif</span>' : '<span class="badge bg-secondary">Pasif</span>' ?></td>
            <td><?= h($r['last_applied_at'] ?? '') ?></td>
            <td class="text-end">
              <div class="btn-group" role="group">
                <a class="btn btn-sm btn-outline-info" href="<?= h(base_url('tenants/health.php?id='.(int)$r['id'])) ?>"><i class="bi bi-heart-pulse"></i> Health</a>
                <a class="btn btn-sm btn-outline-secondary" href="<?= h(base_url('tenants/edit.php?id='.(int)$r['id'])) ?>"><i class="bi bi-pencil"></i> Düzenle</a>
                <a class="btn btn-sm btn-success" href="<?= h(base_url('tenants/apply.php?id='.(int)$r['id'])) ?>"><i class="bi bi-check2-circle"></i> Uygula</a>
              </div>
            </td>
          </tr>
        <?php endforeach; ?>
        <?php if (!$rows): ?>
          <tr><td colspan="11" class="text-center text-muted">Kayıt bulunamadı.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>
<?php include __DIR__ . '/../includes/footer.php'; ?>
