<?php
require_once __DIR__ . '/../includes/auth.php';
$active    = 'tenants';
$pageTitle = 'Toplu İşlemler';

$db   = hub_db();
$c    = hub_config();

// Yalnız aktif tenant'lar
$rows = $db->query("SELECT * FROM tenants WHERE is_active=1 ORDER BY id DESC")->fetchAll();

$action  = $_POST['action'] ?? null;
$ids     = array_map('intval', $_POST['ids'] ?? []);
$results = [];
$errMsg  = '';

// Dinamik: Tenant'tan beklenen modüller
function expected_modules_from_tenant(array $t): array {
    $registry = hub_modules_registry();
    $exp = [];
    foreach ($registry as $cfg) {
        $exp[$cfg['api_field']] = (bool)($t[$cfg['tenant_column']] ?? 0);
    }
    return $exp;
}

// Dinamik: DB'den okunan modüller
function fetch_modules_from_db(array $t): ?array {
    try {
        $dsn = "mysql:host={$t['db_host']};port={$t['db_port']};dbname={$t['db_name']};charset=utf8mb4";
        $pdo = new PDO($dsn, $t['db_user'], $t['db_pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
        
        $registry = hub_modules_registry();
        $configKeys = [];
        foreach ($registry as $cfg) {
            $configKeys[] = $cfg['config_key'];
        }

        $in = implode(",", array_fill(0, count($configKeys), '?'));
        $st = $pdo->prepare("SELECT k,v FROM app_settings WHERE k IN ($in)");
        $st->execute($configKeys);
        
        $map = $st->fetchAll(PDO::FETCH_KEY_PAIR); // k => v
        
        $actual = [];
        foreach ($registry as $cfg) {
            $k = $cfg['config_key'];
            $apiField = $cfg['api_field'];
            $actual[$apiField] = (isset($map[$k]) && (string)$map[$k] === '1');
        }
        return $actual;
    } catch (Throwable $e) {
        return null;
    }
}

// Kısa özet: registry'den rastgele veya ilk 5 tanesini göster
function modules_summary(array $mods): string {
    $registry = hub_modules_registry();
    // İlk 5 modülün keyini al (örnek amaçlı)
    $keys = array_slice(array_keys($registry), 0, 5);
    
    $arr = [];
    foreach ($keys as $k) {
        $apiField = $registry[$k]['api_field'];
        if (isset($mods[$apiField])) {
            $arr[] = strtoupper($k).':'.($mods[$apiField]?'1':'0');
        }
    }
    return implode(' ', $arr);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action && $ids) {
    foreach ($rows as $t) {
        if (!in_array((int)$t['id'], $ids, true)) continue;

        try {
            if ($action === 'ping' || $action === 'health') {
                $exp = expected_modules_from_tenant($t);

                if ($t['apply_method'] === 'api' && !empty($t['api_base_url']) && !empty($t['api_key'])) {
                    // --- API Health/Ping ---
                    $base = rtrim($t['api_base_url'], '/');
                    $url  = ($action === 'ping') ? ($base . '/api/ping.php') : ($base . '/api/health.php');

                    $opts = [
                        'http' => [
                            'method'  => 'GET',
                            'header'  => "X-API-Key: {$t['api_key']}\r\n",
                            'timeout' => 15
                        ]
                    ];
                    $res = @file_get_contents($url, false, stream_context_create($opts));
                    if ($res === false) {
                        $results[] = ['id'=>(int)$t['id'],'name'=>$t['name'],'ok'=>false,'resp'=>'HTTP hata'];
                        continue;
                    }
                    $j = json_decode($res, true);
                    $ok = (bool)($j['ok'] ?? false);

                    $actual = is_array($j['modules'] ?? null) ? array_map('boolval', $j['modules']) : null;
                    $diff = [];
                    if (is_array($actual)) {
                        foreach ($exp as $k=>$v) {
                            if (isset($actual[$k]) && (bool)$actual[$k] !== (bool)$v) {
                                $diff[$k] = ['expected'=>$v, 'actual'=>(bool)$actual[$k]];
                            }
                        }
                    }

                    $results[] = [
                        'id'   => (int)$t['id'],
                        'name' => $t['name'],
                        'ok'   => $ok,
                        'resp' => [
                            'summary' => $actual ? modules_summary($actual) : 'modules:N/A',
                            'diff'    => $diff,
                            'raw'     => is_array($j) ? $j : $res
                        ],
                    ];
                } else {
                    // --- DB Health ---
                    $actual = fetch_modules_from_db($t);
                    if ($actual === null) {
                        $results[] = ['id'=>(int)$t['id'],'name'=>$t['name'],'ok'=>false,'resp'=>'Tenant DB erişimi yok'];
                        continue;
                    }
                    $diff = [];
                    foreach ($exp as $k=>$v) {
                        if ((bool)$actual[$k] !== (bool)$v) {
                            $diff[$k] = ['expected'=>$v, 'actual'=>(bool)$actual[$k]];
                        }
                    }
                    $results[] = [
                        'id'=>(int)$t['id'],
                        'name'=>$t['name'],
                        'ok'=>true,
                        'resp'=>[
                            'summary'=>modules_summary($actual),
                            'diff'=>$diff
                        ]
                    ];
                }
            }
            elseif ($action === 'apply') {
                // --- Apply ---
                if ($t['apply_method'] === 'api') {
                     apply_by_api($t);
                } else {
                     apply_by_db($t);
                }
                $db->prepare("UPDATE tenants SET last_applied_at=NOW() WHERE id=?")->execute([(int)$t['id']]);
                log_tenant((int)$t['id'], 'apply', 'success', 'Bulk apply');

                $results[] = [
                    'id'   => (int)$t['id'],
                    'name' => $t['name'],
                    'ok'   => true,
                    'resp' => 'Ayarlar uygulandı'
                ];
            }
            else {
                $results[] = ['id'=>(int)$t['id'], 'name'=>$t['name'], 'ok'=>false, 'resp'=>'Bilinmeyen işlem'];
            }
        } catch (Throwable $e) {
            if ($action === 'apply') {
                log_tenant((int)$t['id'], 'apply', 'error', $e->getMessage());
            }
            $results[] = ['id'=>(int)$t['id'], 'name'=>$t['name'], 'ok'=>false, 'resp'=>$e->getMessage()];
        }
    }
}
include __DIR__ . '/../includes/header.php';
?>
<div class="d-flex justify-content-between align-items-center mb-3">
  <h3 class="mb-0">Toplu İşlemler</h3>
  <a class="btn btn-secondary" href="<?= h(base_url('tenants/index.php')) ?>"><i class="bi bi-arrow-left"></i> Geri</a>
</div>
<?php if ($errMsg): ?>
  <div class="alert alert-danger"><?= h($errMsg) ?></div>
<?php endif; ?>

<form method="post" class="card mb-3">
  <div class="card-body">
    <div class="row g-2 align-items-center">
      <div class="col-auto"><strong>İşlem:</strong></div>
      <div class="col-auto">
        <select name="action" class="form-select">
          <option value="ping"   <?= ($action==='ping')   ? 'selected' : '' ?>>Ping (API)</option>
          <option value="health" <?= ($action==='health') ? 'selected' : '' ?>>Health (API/DB)</option>
          <option value="apply"  <?= ($action==='apply')  ? 'selected' : '' ?>>Apply (Ayarları Uygula)</option>
        </select>
      </div>
      <div class="col-auto">
        <button class="btn btn-primary"><i class="bi bi-play-fill"></i> Çalıştır</button>
      </div>
    </div>
    <hr>
    <div class="table-responsive">
      <table class="table table-bordered align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th style="width:42px">
              <input type="checkbox" onclick="document.querySelectorAll('.sel').forEach(cb=>cb.checked=this.checked)">
            </th>
            <th>#</th>
            <th>Ad</th>
            <th>Domain</th>
            <th>Yöntem</th>
            <th>API URL</th>
            <th>Aktif</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $t): ?>
            <tr>
              <td><input type="checkbox" class="sel" name="ids[]" value="<?= (int)$t['id'] ?>"></td>
              <td><?= (int)$t['id'] ?></td>
              <td><?= h($t['name']) ?></td>
              <td><?= h($t['domain']) ?></td>
              <td><span class="badge bg-secondary"><?= h($t['apply_method']) ?></span></td>
              <td class="text-truncate" style="max-width:280px"><?= h($t['api_base_url'] ?: '-') ?></td>
              <td><?= (int)$t['is_active'] ? '<span class="badge bg-success">Aktif</span>' : '<span class="badge bg-secondary">Pasif</span>' ?></td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?>
            <tr><td colspan="7" class="text-center text-muted">Aktif tenant yok.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</form>

<?php if ($action): ?>
  <div class="card">
    <div class="card-header">
      <strong>Sonuçlar</strong> <small class="text-muted">(<?= h($action) ?>)</small>
    </div>
    <div class="card-body">
      <?php if (!$results): ?>
        <div class="text-muted">Seçili tenant yok.</div>
      <?php else: ?>
        <?php foreach ($results as $r): ?>
          <div class="mb-3 p-2 border rounded">
            <div class="d-flex justify-content-between">
              <div><strong>#<?= (int)$r['id'] ?> - <?= h($r['name']) ?></strong></div>
              <div><?= $r['ok'] ? '<span class="badge bg-success">OK</span>' : '<span class="badge bg-danger">FAIL</span>' ?></div>
            </div>
            <pre class="bg-light p-2 mt-2" style="white-space:pre-wrap"><?= h(is_string($r['resp']) ? $r['resp'] : json_encode($r['resp'], JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE)) ?></pre>
          </div>
        <?php endforeach; ?>
      <?php endif; ?>
    </div>
  </div>
<?php endif; ?>
<?php include __DIR__ . '/../includes/footer.php'; ?>