<?php
// AdminHub -> includes/functions.php
require_once __DIR__ . '/../config/database.php';
// Modül kayıt dosyasını dahil et
require_once __DIR__ . '/../modules_registry.php';

if (session_status() === PHP_SESSION_NONE) { session_start(); }

/* ---------- Genel yardımcılar ---------- */
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function base_url(string $path=''): string {
    $c = hub_config();
    return rtrim($c['BASE_URL'],'/') . '/' . ltrim($path,'/');
}

/* ---------- CSRF ---------- */
function csrf_token(): string {
    if (empty($_SESSION['csrf'])) {
        $_SESSION['csrf'] = bin2hex(random_bytes(16));
    }
    return $_SESSION['csrf'];
}
function csrf_check(?string $t): bool {
    return $t && isset($_SESSION['csrf']) && hash_equals($_SESSION['csrf'], $t);
}

/* ---------- Flash ---------- */
function flash_set(string $type, string $msg): void {
    $_SESSION['flash'][$type] = $msg;
}
function flash_get(?string $type=null): ?string {
    if ($type===null) return null;
    $msg = $_SESSION['flash'][$type] ?? null;
    if ($msg!==null) unset($_SESSION['flash'][$type]);
    return $msg;
}

/* ---------- API Key üretimi ---------- */
function generate_api_key(): string {
    $raw = random_bytes(32);
    $b64 = rtrim(strtr(base64_encode($raw), '+/', '-_'), '=');
    return 'sk-' . $b64;
}

/* ---------- Admin Auth ---------- */
function current_admin(): ?array { return $_SESSION['hub_admin'] ?? null; }
function is_admin(): bool { return current_admin() !== null; }

function login_blocked(string $ip): bool {
    $c = hub_config();
    $max = (int)($c['RATE_LIMIT_MAX_FAIL'] ?? 5);
    $st = hub_db()->prepare("SELECT COUNT(*) AS c FROM login_attempts WHERE ip=? AND ok=0 AND attempted_at >= (NOW() - INTERVAL 10 MINUTE)");
    $st->execute([$ip]);
    $row = $st->fetch();
    return ((int)$row['c']) >= $max;
}

function log_login_attempt(string $ip, string $email, bool $ok): void {
    $st = hub_db()->prepare("INSERT INTO login_attempts (ip,email,ok) VALUES (?,?,?)");
    $st->execute([$ip, $email, $ok?1:0]);
}

function attempt_login(string $email, string $pass): bool {
    $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    if (login_blocked($ip)) { return false; }

    $st = hub_db()->prepare("SELECT * FROM admins WHERE email=? AND is_active=1 LIMIT 1");
    $st->execute([$email]);
    $adm = $st->fetch();

    $ok = $adm && password_verify($pass, $adm['password_hash']);
    log_login_attempt($ip, $email, $ok);

    if ($ok) {
        session_regenerate_id(true);
        $_SESSION['hub_admin'] = [
            'id'    => (int)$adm['id'],
            'email' => $adm['email'],
            'name'  => $adm['full_name'],
        ];
        hub_db()->prepare("UPDATE admins SET last_login_at=NOW() WHERE id=?")->execute([$adm['id']]);
        return true;
    }
    return false;
}

function require_admin(): void {
    if (!is_admin()) {
        header('Location: ' . base_url('auth/login.php'));
        exit;
    }
}

/* ---------- Tenant yardımcıları ---------- */
function get_tenant(int $id): ?array {
    $st = hub_db()->prepare("SELECT * FROM tenants WHERE id=?");
    $st->execute([$id]);
    return $st->fetch() ?: null;
}

function log_tenant(int $tenantId, string $action, string $status, ?string $msg=null): void {
    $st = hub_db()->prepare("INSERT INTO tenant_logs (tenant_id,action,status,message) VALUES (?,?,?,?)");
    $st->execute([$tenantId, $action, $status, $msg]);
}

/* ---------- app_settings upsert (tenant DB’de) ---------- */
function upsert_app_setting(PDO $pdo, string $k, ?string $v): void {
    $pdo->prepare("INSERT INTO app_settings (k,v) VALUES (?,?) ON DUPLICATE KEY UPDATE v=VALUES(v)")
        ->execute([$k, $v]);
}

/* ---------- Uygulama yöntemleri ---------- */
/**
 * DB üstünden uygula (tenant veritabanına doğrudan yaz)
 * Registry'den dinamik olarak modülleri okur.
 */
function apply_by_db(array $tenant): void {
    $dsn = "mysql:host={$tenant['db_host']};port={$tenant['db_port']};dbname={$tenant['db_name']};charset=utf8mb4";
    $pdo = new PDO($dsn, $tenant['db_user'], $tenant['db_pass'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);

    // 1) Modülleri Registry'den al
    $registry = hub_modules_registry();
    $map = [];

    foreach ($registry as $key => $cfg) {
        $col = $cfg['tenant_column'];
        $confKey = $cfg['config_key'];
        // Tenant tablosundaki değeri al (0 veya 1), string olarak ayarla
        $map[$confKey] = (string)((int)($tenant[$col] ?? 0));
    }

    // Abonelik bilgileri
    $map['SUBSCRIPTION_TYPE'] = $tenant['subscription_type'] ?? 'free';
    $map['SUBSCRIPTION_END']  = $tenant['subscription_end'] ?: null;

    // 2) Ödeme anahtarları (PAY_*)
    $pay = [
      'PAY_ENABLED'    => (string)((int)($tenant['pay_enabled'] ?? 0)),
      'PAY_PROVIDER'   => $tenant['pay_provider'] ?? 'none',
      'PAY_TRIAL_MODE' => (string)((int)($tenant['pay_trial_mode'] ?? 1)),
      'PAY_CURRENCY'   => $tenant['pay_currency'] ?? 'TRY',
      'PAY_IYZICO_API_KEY' => $tenant['pay_iyzico_api_key'] ?? null,
      'PAY_IYZICO_SECRET'  => $tenant['pay_iyzico_secret']    ?? null,
      'PAY_STRIPE_PK' => $tenant['pay_stripe_pk'] ?? null,
      'PAY_STRIPE_SK' => $tenant['pay_stripe_sk'] ?? null,
      'PAY_STRIPE_WEBHOOK_SECRET' => $tenant['pay_stripe_webhook_secret'] ?? null,
      'PAY_SUCCESS_URL' => $tenant['pay_success_url'] ?? null,
      'PAY_CANCEL_URL'  => $tenant['pay_cancel_url']  ?? null,
      'PAY_RETURN_URL'  => $tenant['pay_return_url']  ?? null,
    ];

    foreach (array_merge($map, $pay) as $k=>$v) {
        upsert_app_setting($pdo, $k, $v);
    }

    // 2.5) Fiyatlar (PLAN_PRICES -> app_settings)
    $c  = hub_config();
    $pp = $c['PLAN_PRICES'] ?? [];
    $basicMonthly = (float)($pp['basic']['monthly'] ?? 50);
    $basicYearly  = (float)($pp['basic']['yearly']  ?? ($basicMonthly * 12));
    $proMonthly   = (float)($pp['pro']['monthly']   ?? 100);
    $proYearly    = (float)($pp['pro']['yearly']    ?? ($proMonthly * 12));

    upsert_app_setting($pdo, 'PRICE_CURRENCY', (string)($pp['currency'] ?? 'TRY'));
    upsert_app_setting($pdo, 'BASIC_PRICE',    (string)$basicMonthly);
    upsert_app_setting($pdo, 'BASIC_YEARLY',   (string)$basicYearly);
    upsert_app_setting($pdo, 'PRO_PRICE',      (string)$proMonthly);
    upsert_app_setting($pdo, 'PRO_YEARLY',     (string)$proYearly);

    // 3) API anahtarlarını tenant DB'ye yaz
    upsert_app_setting($pdo, 'API_ENABLED', '1');
    if (!empty($tenant['api_key'])) {
        upsert_app_setting($pdo, 'API_KEYS', $tenant['api_key']);
    }

    // 4) users abonelik uyarlama
    $st = $pdo->prepare("UPDATE users SET subscription_type = ?, subscription_end = ? WHERE is_active = 1");
    $st->execute([$tenant['subscription_type'] ?? 'free', $tenant['subscription_end'] ?: null]);
}

/**
 * API üstünden uygula (tenant/api/set_config.php)
 * Registry'den dinamik olarak modülleri hazırlar.
 */
function apply_by_api(array $tenant): void {
    if (empty($tenant['api_base_url']) || empty($tenant['api_key'])) {
        throw new RuntimeException('API bilgileri eksik.');
    }

    $url = rtrim($tenant['api_base_url'],'/') . '/api/set_config.php';

    // Modülleri Registry'den dinamik oluştur
    $registry = hub_modules_registry();
    $modulesPayload = [];
    
    foreach ($registry as $key => $cfg) {
        $apiField = $cfg['api_field']; // 'invoices', 'crm' vs.
        $col = $cfg['tenant_column'];
        $modulesPayload[$apiField] = (bool)($tenant[$col] ?? 0);
    }

    // Payment (PAY_*)
    $payment = [
        'enabled'     => (bool)($tenant['pay_enabled'] ?? 0),
        'provider'    => $tenant['pay_provider'] ?? 'none',
        'trial_mode'  => (bool)($tenant['pay_trial_mode'] ?? 1),
        'currency'    => $tenant['pay_currency'] ?? 'TRY',
        'iyzico_api_key' => $tenant['pay_iyzico_api_key'] ?? null,
        'iyzico_secret'  => $tenant['pay_iyzico_secret']    ?? null,
        'stripe_pk' => $tenant['pay_stripe_pk'] ?? null,
        'stripe_sk' => $tenant['pay_stripe_sk'] ?? null,
        'stripe_webhook_secret' => $tenant['pay_stripe_webhook_secret'] ?? null,
        'success_url' => $tenant['pay_success_url'] ?? null,
        'cancel_url'  => $tenant['pay_cancel_url']  ?? null,
        'return_url'  => $tenant['pay_return_url']  ?? null,
    ];

    // Pricing
    $c  = hub_config();
    $pp = $c['PLAN_PRICES'] ?? [];
    $pricing = [
        'currency' => (string)($pp['currency'] ?? 'TRY'),
        'basic' => [
            'monthly' => (float)($pp['basic']['monthly'] ?? 50),
            'yearly'  => (float)($pp['basic']['yearly']  ?? (($pp['basic']['monthly'] ?? 50) * 12)),
        ],
        'pro' => [
            'monthly' => (float)($pp['pro']['monthly'] ?? 100),
            'yearly'  => (float)($pp['pro']['yearly']  ?? (($pp['pro']['monthly'] ?? 100) * 12)),
        ],
    ];

    $payload = [
        'subscription_type' => $tenant['subscription_type'] ?? 'free',
        'subscription_end'  => $tenant['subscription_end'] ?: null,
        'modules'           => $modulesPayload,
        'payment'           => $payment,
        'pricing'           => $pricing,
    ];

    $opts = [
        'http' => [
            'method'  => 'POST',
            'header'  => "Content-Type: application/json\r\nX-API-Key: {$tenant['api_key']}\r\n",
            'content' => json_encode($payload, JSON_UNESCAPED_UNICODE),
            'timeout' => 20,
        ]
    ];

    $res = @file_get_contents($url, false, stream_context_create($opts));
    if ($res === false) {
        throw new RuntimeException('API çağrısı başarısız.');
    }
    $j = json_decode($res, true);
    if (!is_array($j) || ($j['ok'] ?? false) !== true) {
        throw new RuntimeException('API hata: ' . ($j['error'] ?? 'bilinmiyor'));
    }
}
?>